import json
import base64

from Crypto.Hash import SHA256
from Crypto.PublicKey import RSA
from Crypto.Signature.pkcs1_15 import PKCS115_SigScheme

from fastapi import FastAPI, Request, Response
from uvicorn import run
from config import public_key_file_path
from helper import normalize_message

app = FastAPI()

with open(public_key_file_path, "rb") as f:
    encoded_pub_key = base64.urlsafe_b64encode(
        RSA.importKey(f.read()).export_key(),
    ).decode('utf-8')


@app.route("/callback", methods=["POST", "GET"])
async def validate_request(
        request: Request,
):
    _headers = request.headers
    body = await request.body()

    print(body.decode())

    if not body:
        payload = {}
    else:
        payload = json.loads(body.decode("utf-8"))

    joined_result = normalize_message(payload).encode("utf-8")

    message = "{}{}".format(
        base64.urlsafe_b64encode(joined_result).decode("utf-8"),
        str(
            _headers.get("x-access-timestamp"),
        ),
    )
    _headers_access_token = _headers.get("x-access-token", "")
    if not _headers_access_token:
        raise ValueError("Access token is not found in the headers.")

    _headers_access_token = base64.urlsafe_b64decode(_headers_access_token.encode())

    headers_access_token = base64.urlsafe_b64encode(
        RSA.importKey(_headers_access_token).export_key(),
    ).decode('utf-8')

    if headers_access_token != encoded_pub_key:
        raise ValueError("Invalid public key")

    signature = _headers.get("x-access-signature")

    pub_key = base64.urlsafe_b64decode(
        encoded_pub_key.encode("utf-8"),
    )
    pub_key_rsa = RSA.importKey(pub_key)
    hash_of_received = SHA256.new(message.encode("utf-8"))
    verifier = PKCS115_SigScheme(pub_key_rsa)
    try:
        base64_bytes = signature.encode("ascii")
        signature_bytes = base64.urlsafe_b64decode(base64_bytes)
        verifier.verify(hash_of_received, signature_bytes)
    except ValueError:
        print("Invalid signature")
        raise ValueError("Invalid signature")

    print("Signature is valid")

    return Response(status_code=200)


if __name__ == '__main__':
    run(app, host="0.0.0.0", port=8000)
