import express from "express";
import crypto from "crypto";
import fs from "fs";
import base64url from "base64url";
import {publicKeyFilePath} from "./config.js";



const app = express();
const PORT = 8000;

const publicKey = fs.readFileSync(publicKeyFilePath, "utf8").trim()
const encodedPubKey = base64url.encode(publicKey);

app.use(express.json());


function parseJson(prefix, obj, result) {
  if (typeof obj === "object" && obj !== null) {
    if (Array.isArray(obj)) {
      obj.forEach((item, index) =>
        parseJson(`${prefix ? prefix + ":" : ""}${index}`, item, result)
      );
    } else {
      Object.keys(obj).forEach((key) =>
        parseJson(`${prefix ? prefix + ":" : ""}${key}`, obj[key], result)
      );
    }
  } else {
    result.push(`${prefix}:${obj ?? "None"}`);
  }
}

function normalizePayload(payload) {
  const result = [];
  parseJson("", payload, result);
  result.sort();
  return result.join(";");
}

function verifySignature(signature, normalizedPayload, timestamp, publicKey) {
  const dataToVerify =
    Buffer.from(normalizedPayload).toString("base64") + timestamp;
  const verifier = crypto.createVerify("RSA-SHA256");
  verifier.update(dataToVerify);
  return verifier.verify(publicKey, signature, "base64");
}

app.all("/callback/:status", (req, res) => {
  const {
    ["x-access-token"]: accessToken,
    ["x-access-signature"]: signature,
    ["x-access-timestamp"]: timestamp,
  } = req.headers;

  console.log(req.body)

  const newDecodeToken = base64url.decode(accessToken, "utf8").trim();

  const newEncodeAccessToken = base64url.encode(newDecodeToken)

  if (newEncodeAccessToken !== encodedPubKey) {
    console.log("Invalid public key");
    return res.status(400).send("Invalid public key");
  }

  if (!signature || !timestamp) {
    console.log("Missing signature or timestamp");
    return res.status(400).send("Missing signature or timestamp");
  }

  const normalizedPayload = normalizePayload(req.body);
  const isValid = verifySignature(
    signature,
    normalizedPayload,
    timestamp,
    publicKey
  );

  if (!isValid) {
    console.log("Signature is not valid");
    return res.status(400).send("Invalid signature");
  }

  console.log("Signature is valid");

  res.sendStatus(200);
});

app.listen(PORT, () => {
  console.log(`Server running on http://0.0.0.0:8000`);
});
